<?php
require_once 'config.php';
require_once 'header.php';

// Redirect if not logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$user_id = $_SESSION['user_id'];

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_entry'])) {
    $full_name = trim($_POST['full_name'] ?? '');
    $college = trim($_POST['college'] ?? '');
    $enrollment_no = trim($_POST['enrollment_no'] ?? '');
    $event_name = trim($_POST['event_name'] ?? '');
    $event_date = $_POST['event_date'] ?? '';
    $description = trim($_POST['description'] ?? '');
    $hours_served = $_POST['hours_served'] ?? '';
    $certificate_required = $_POST['certificate_required'] ?? 'no';

    if (empty($full_name) || empty($college) || empty($enrollment_no) || empty($event_name) || empty($event_date) || empty($description) || empty($hours_served)) {
        $error = 'All fields are required.';
    } elseif (!is_numeric($hours_served) || $hours_served <= 0) {
        $error = 'Hours served must be a positive number.';
    } else {
        $stmt = $pdo->prepare("INSERT INTO nss_entries (user_id, full_name, college, enrollment_no, event_name, event_date, description, hours_served, certificate_required) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
        if ($stmt->execute([$user_id, $full_name, $college, $enrollment_no, $event_name, $event_date, $description, $hours_served, $certificate_required])) {
            $success = 'NSS activity recorded successfully.';
        } else {
            $error = 'Submission failed. Please try again.';
        }
    }
}

// Fetch user's NSS entries
$stmt = $pdo->prepare("SELECT * FROM nss_entries WHERE user_id = ? ORDER BY created_at DESC");
$stmt->execute([$user_id]);
$entries = $stmt->fetchAll();
?>

<h2>Welcome, <?= htmlspecialchars($_SESSION['user_name']) ?></h2>

<h3>Record New NSS Activity</h3>
<?php if (isset($error)): ?><p class="error"><?= htmlspecialchars($error) ?></p><?php endif; ?>
<?php if (isset($success)): ?><p class="success"><?= htmlspecialchars($success) ?></p><?php endif; ?>

<form method="post">
    <label for="full_name">Full Name:</label>
    <input type="text" name="full_name" id="full_name" required value="<?= htmlspecialchars($_POST['full_name'] ?? $_SESSION['user_name']) ?>">

    <label for="college">College/Institution:</label>
    <input type="text" name="college" id="college" required value="<?= htmlspecialchars($_POST['college'] ?? '') ?>">

    <label for="enrollment_no">Enrollment Number:</label>
    <input type="text" name="enrollment_no" id="enrollment_no" required value="<?= htmlspecialchars($_POST['enrollment_no'] ?? '') ?>">

    <label for="event_name">Event/Activity Name:</label>
    <input type="text" name="event_name" id="event_name" required value="<?= htmlspecialchars($_POST['event_name'] ?? '') ?>">

    <label for="event_date">Event Date:</label>
    <input type="date" name="event_date" id="event_date" required value="<?= htmlspecialchars($_POST['event_date'] ?? '') ?>">

    <label for="description">Description of Work Done:</label>
    <textarea name="description" id="description" rows="5" required><?= htmlspecialchars($_POST['description'] ?? '') ?></textarea>

    <label for="hours_served">Hours Served:</label>
    <input type="number" step="0.5" name="hours_served" id="hours_served" required value="<?= htmlspecialchars($_POST['hours_served'] ?? '') ?>">

    <label for="certificate_required">Certificate Required?</label>
    <select name="certificate_required" id="certificate_required">
        <option value="no" <?= (isset($_POST['certificate_required']) && $_POST['certificate_required'] == 'no') ? 'selected' : '' ?>>No</option>
        <option value="yes" <?= (isset($_POST['certificate_required']) && $_POST['certificate_required'] == 'yes') ? 'selected' : '' ?>>Yes</option>
    </select>

    <button type="submit" name="submit_entry">Submit Activity</button>
</form>

<h3>Your Past NSS Activities</h3>
<?php if (count($entries) > 0): ?>
    <table>
        <thead>
            <tr>
                <th>ID</th>
                <th>Event</th>
                <th>Date</th>
                <th>Hours</th>
                <th>Certificate</th>
                <th>Status</th>
            </tr>
        </thead>
        <tbody>
        <?php foreach ($entries as $entry): ?>
            <tr>
                <td><?= $entry['id'] ?></td>
                <td><?= htmlspecialchars($entry['event_name']) ?></td>
                <td><?= date('d M Y', strtotime($entry['event_date'])) ?></td>
                <td><?= $entry['hours_served'] ?></td>
                <td><?= ucfirst($entry['certificate_required']) ?></td>
                <td><?= ucfirst($entry['status']) ?></td>
            </tr>
        <?php endforeach; ?>
        </tbody>
    </table>
<?php else: ?>
    <p>You haven't recorded any NSS activities yet.</p>
<?php endif; ?>

<?php require_once 'footer.php'; ?>

